using System;
using System.Xml;

using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary.Messages
{
	/// <summary>
	/// This class represents VistALink setup and introductory text 
	/// response security message as defined in secSetupIntroResponse.xsd.
	/// </summary>
	public sealed class SecuritySetupAndIntroResponseMessage : SecurityBaseResponseMessage 
	{
		private readonly RemoteServerSetupAndIntroInfo _serverInfo;

		/// <summary>
		/// VistALink message XSD schema file name string. 		
		/// This constant is made public to recognize message XML.
		/// </summary>
		public const string XsdName = "secSetupIntroResponse.xsd";

		/// <summary>
		/// VistALink message response type string. 
		/// This constant is made public to recognize message XML.
		/// </summary>
		public const string ResponseType = "AV.SetupAndIntroText";

		/// <summary>
		/// The only constructor accepting server setup and intro data. 
		/// </summary>
		/// <param name="serverInfo">Server setup and intro data. Required.</param>
		public SecuritySetupAndIntroResponseMessage( RemoteServerSetupAndIntroInfo serverInfo ) 
			: base( XsdName, ResponseType, ResponseStatus.Success ) 
		{
			if( serverInfo == null )
				throw( new ArgumentNullException( "serverInfo" ) );

			_serverInfo = serverInfo;
		}

		/// <summary>
		/// Standard VistALink message deserialization constructor. 
		/// Allows recreating message object from XML document.
		/// </summary>
		/// <param name="sourceDoc">Source XML document.</param>
		public SecuritySetupAndIntroResponseMessage( XmlDocument sourceDoc )
			: base( XsdName, ResponseType, sourceDoc ) 
		{
			if( _responseStatus != ResponseStatus.Success )
				throw( new XmlParseException( SR.Exceptions.SecSetupAndInroInvalidResponseStatus( _responseStatus ) ) );

			_serverInfo = RemoteServerSetupAndIntroInfo.Parse( XmlParseGetResponseElement( sourceDoc ) );
		}

		///<summary>
		/// Overriden method adding message-specific content to response element 
		/// of XML message representation during XML serialization. 
		/// </summary>
		/// <param name="writer">XmlWriter to use.</param>
		protected override void WriteXmlResponseNodeContent( XmlWriter writer )
		{			
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			_serverInfo.WriteSetupAndIntroInfoToXml( writer );
		}

		/// <summary>
		/// Remote server setup and intro text data structure.
		/// </summary>
		public RemoteServerSetupAndIntroInfo ServerInfo
		{
			get
			{
				return _serverInfo;
			}
		}
	}
}
